﻿using System;
using System.Collections;
using System.Globalization;
using System.Net;
using System.Reflection;
using System.Text;
using System.Threading;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Server;
using gov.va.med.VBECS.HL7Server.Core;
using gov.va.med.vbecs.Common.AppServices;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;
using IServer = gov.va.med.vbecs.Common.IServer;
using ServicesGlobalContext = VBECS.Services.Common.AppServices.GlobalContext;

namespace gov.va.med.VBECS.HL7Server
{
    /// <summary>
    /// Server side object responsible for incoming messages processing
    /// </summary>
    internal class Server : IServer
    {
        #region Variables

        private const string UNSUPPORTED_INTERFACE_ERROR = "The Sending and/or Receiving Application(s) in the message are not supported by VBECS in this combination.\n\n";

        #endregion

        public event EventHandler<ThreadExceptionEventArgs> FatalErrorOccured;
        // Internal TCP server
        private Communication.Server.IServer _server;
        // Logger
        private readonly ILogger _logger =
            LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);
        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

        public void Start()
        {
            if (_server != null) return; // The server already started

            // Read port number from configuration
            var portNumber =
                (int)GlobalContext.Instance().AppSettingsReader.GetValue("ListenerPortNumber", typeof(int));
            _server = ServerFactory.CreateServer(new IPEndPoint(IPAddress.Any, portNumber));

            _server.ClientDisconnected += server_client_disconnected;
            _server.ClientConnected += server_client_connected;
            _server.Start();
            _logger.Debug(string.Format("A server started (on port {0})", portNumber));
        }

        public void Stop()
        {
            if (_server == null) return; // The server is not started
            _server.Stop();
            _server = null;
        }

        private void server_client_connected(object sender, ClientDummyEventArgs e)
        {
            _logger.Debug("A new client is connected. Client Id = " + e.Client.Id);

            //Register to MessageReceived event to receive messages from new client
            e.Client.MessageReceived += client_message_received;
        }

        private void server_client_disconnected(object sender, ClientDummyEventArgs e)
        {
            _logger.Debug("A client is disconnected! Client Id = " + e.Client.Id);
        }

        private void client_message_received(object sender, MessageEventArgs e)
        {
            //Server only accepts RawMessages
            if (!(e.Message is RawDataMessage))
            {
                _logger.Warn(string.Format("Client sent unknown message: [type={0}]", (e.Message == null ? "null" : e.Message.GetType().ToString())));
                return;
            }
            //construct text message
            var message = new TextMessage(((RawDataMessage)e.Message).Data);

            //Get a reference to the client
            var client = (IClientDummy)sender;
            _logger.Debug("Client sent a message: " + new TextMessage(message.Text).Text +
                          " (Client Id = " + client.Id + ")");

            var errorRetrievingInterface = false;
            var isNonVbecsPatientUpdate = false;
            var isBceUpdate = false; // CR 2940
            HL7ProtocolMessage ackMessage = null;
            HL7Interface hl7Interface = null;

            try
            {
                hl7Interface = get_interface_parameters(message.Text, ref isNonVbecsPatientUpdate, ref isBceUpdate, ref errorRetrievingInterface); 
                if (errorRetrievingInterface || (hl7Interface == null && !isNonVbecsPatientUpdate))
                {
                    // Should never get here unless unsupported interface sends us a message
                    // or interface definition on sending system is incorrect.
                    // Addresses CR 2140, CR 2940
                    ackMessage = HandleUnsupportedInterface(message.Text, client.EndPoint.Address.ToString());
                }
                else if (hl7Interface == null)
                {
                    ackMessage = HL7AckMessage.BuildAckMessage(message.Text, AckCodes.AA, string.Empty);
                }
                else if (!(isBceUpdate && !hl7Interface.InterfaceActiveIndicator)) // CR 2940
                {
                    // CR 2606, CR 2659
                    var errorMessageTable = ValidateVbecsRequiredMessageContent(message.Text);
                    // CR 3349: added call to client.EndPoint.Address
                    ackMessage = errorMessageTable == null ? SimpleListenerHL7Manager.ProcessIncomingHL7Message(hl7Interface, message.Text) : HandleUnsupportedPatientName(hl7Interface, message.Text, errorMessageTable, client.EndPoint.Address.ToString());
                }
                //
                if (ackMessage != null)
                {
                    //Send reply message to the client
                    client.Send(new RawDataMessage(ackMessage.ToByteArrayWithoutControlCharacters()));
                    if ((hl7Interface != null && hl7Interface.LogEvents) || hl7Interface == null)
                    {
                        // EventLogAppender
                        //CR 3509
                        if (ServicesGlobalContext.Instance() != null)
                        {
                            if (ServicesGlobalContext.Instance().ServiceProperties != null)
                            {
                                if (ServicesGlobalContext.Instance().ServiceProperties.Name != String.Empty)
                                {
                                    if (ackMessage != null)
                                    {
                                        if (ackMessage.GetMessage().Length > 0)
                                        {
                                            if (ackMessage.GetMessage() != string.Empty)
                                            {
                                                _eventsLogger.Info(string.Concat("Response Message: \n\n\n",
                                                    ServicesGlobalContext.Instance().ServiceProperties.Name,
                                                    " sent message:\n\n", ackMessage.GetMessage()));
                                            }
                                        }
                                    }
                                }
                            }
                            
                        }
                    }
                }		
            }
            catch (Exception xcp)
            {
                //Log exception but continue listening for other messages.
                _logger.Error(string.Format("Exception occurred while processing client request. Message:\r\n{0}", message.Text), xcp);
                _eventsLogger.Error(string.Format("Exception occurred while processing client request. Message:\r\n{0}", message.Text), xcp);

                if (hl7Interface == null) return;
                // Only send mail message if interface parameters are defined to prevent exception.
                // Related to CR 2140.
                var exceptionMessage = vbecs.ExceptionManagement.ExceptionManager.FormatExceptionInfoAndDumpToString(xcp);
                send_mail_message(exceptionMessage, hl7Interface);
            }
        }

        private HL7Interface get_interface_parameters(string message, ref bool isNonVbecsPatientUpdate, ref bool isBceUpdate, ref bool errorRetrievingInterface)
        {
            HL7Interface hl7Interface = null;
            //
            try
            {
                if (message == null)
                {
                    throw (new ArgumentNullException("message"));
                }
                //
                // This includes field separator in position [0] and encoding characters in [1] - [4]
                var delimiters = HL7Utility.ParseGetMessageDelimiters(message);
                var mshSegment = HL7Utility.ParseGetRequiredMessageSegment(message, SegmentTypeNames.MSH);
                var interfaceAppId = HL7Utility.ParseGetSegmentData(mshSegment, delimiters[0].ToString(CultureInfo.InvariantCulture), 2);
                var vbecsAppId = HL7Utility.ParseGetSegmentData(mshSegment, delimiters[0].ToString(CultureInfo.InvariantCulture), 4);
                //
                if (interfaceAppId != null && vbecsAppId != null)
                {
                    hl7Interface = HL7Interface.GetInterfaceByApplicationIdentifiers(interfaceAppId, vbecsAppId);
                    if (hl7Interface != null)
                    {
                        if (hl7Interface.InterfaceName == InterfaceName.Patient_Update.ToString())
                        {
                            var pidSegment = HL7Utility.ParseGetRequiredMessageSegment(message, SegmentTypeNames.PID);
                            //
                            // CR 2971
                            var vistAPatientId = HL7Utility.ParseGetVistAPatientId(pidSegment, delimiters);
                            //
                            if (!PatientChange.IsVbecsPatient(vistAPatientId))
                            {
                                isNonVbecsPatientUpdate = true;
                                //
                                hl7Interface = null;
                            }
                        }
                        else if (hl7Interface.InterfaceName == InterfaceName.BCE_COTS.ToString())
                        {
                            // CR 2940
                            isBceUpdate = true;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                errorRetrievingInterface = true;
                // EventLogAppender
                // EventsLoggerSingleton.Instance.Log(HL7Utility.GetInnerExceptionFromException(ex), EventLogEntryType.Error);
                _eventsLogger.Error(HL7Utility.GetInnerExceptionFromException(ex));
            }
            //
            return hl7Interface;
        }

        /// <summary>
        /// CR 2606
        /// </summary>
        private HL7ProtocolMessage HandleUnsupportedInterface(string hl7Message, string ipAddress)
        {
            // 
            const string errorMesage = "Incorrect or unsupported key identifiers in message. Check HL7 Application Parameters.";
            //
            HL7ProtocolMessage ackMessage = HL7AckMessage.BuildAckMessage(hl7Message, AckCodes.AR, errorMesage);
            //
            var warningMsg = send_unsupported_Hl7_nessage_error_details(ipAddress, hl7Message, UNSUPPORTED_INTERFACE_ERROR, null);
            // EventLogAppender
            // Record full error details including HL7 message contents in the event log
            _eventsLogger.Warn(warningMsg);

            //
            return ackMessage;
        }

        /// <summary>
        /// Builds and sends an email message and contracts a warning that an unsupported HL7 message was received.
        /// The email is constructed without the full HL7 details; we use the MSH segment as the "ID" so that the recipient can tie 
        /// the email message to the full HL7 message, which is recorded in the event log.  
        /// CR 2606, CR 2898
        /// </summary>
        private string send_unsupported_Hl7_nessage_error_details(
            string ipAddress, string hl7Message, string errorMessage, HL7Interface hl7Interface)
        {
            var fullErrorMessageBuilder = new StringBuilder();
            //
            fullErrorMessageBuilder.Append("The ");
            fullErrorMessageBuilder.Append(ServicesGlobalContext.Instance().ServiceProperties.Name);
            fullErrorMessageBuilder.Append(" Windows Service received an unsupported HL7 message from IP Address ");
            fullErrorMessageBuilder.Append(ipAddress);
            fullErrorMessageBuilder.Append(".\n\n");
            fullErrorMessageBuilder.Append(errorMessage);
            //
            // Build email message without the full HL7 details
            var emailMessageBuilder = new StringBuilder(fullErrorMessageBuilder.ToString());
            emailMessageBuilder.Append("\n\nThe Message ID is: ");
            emailMessageBuilder.Append(HL7Utility.ParseGetRequiredMessageSegment(hl7Message, SegmentTypeNames.MSH));
            emailMessageBuilder.Append("\n\nPlease check the server event log for the full HL7 message details.");
            if (hl7Interface != null)
                send_mail_message(emailMessageBuilder.ToString(), hl7Interface);

            fullErrorMessageBuilder.Append("\n\nMessage Received:\n\n");
            fullErrorMessageBuilder.Append(hl7Message);
            return fullErrorMessageBuilder.ToString();
        }

        /// <summary>
        /// send_mail_message
        /// </summary>
        private void send_mail_message(string messageBody, HL7Interface hl7Interface)
        {
            try
            {
                var mail = new HL7MailMessage();
                //
                mail.SendMessage(hl7Interface.InterfaceAdministratorEmail, "VBECS.HL7@domain.ext", string.Concat("Error in VBECS HL7 Listener for ", hl7Interface.InterfaceName), messageBody, "mail.server");
            }
            catch (Exception err)
            {
                var exceptionMessage = vbecs.ExceptionManagement.ExceptionManager.FormatExceptionInfoAndDumpToString(err);
                // EventLogAppender
                _eventsLogger.Error(exceptionMessage);
            }
        }

        /// <summary>
        /// CR 2659: New method to verify VBECS required field data across all message types
        /// 
        /// Supported Validation
        /// ---------------------------------------------------------------------------------------
        /// (1) Patient name: the name fields found in the message must meet VBECS guidelines for 
        ///		maximum length and presence.  See the VBECS Application Interface Specification for 
        ///		details.
        ///		
        /// </summary>
        private static Hashtable ValidateVbecsRequiredMessageContent(string hl7Message)
        {
            const string pidSegmentId = "PID";
            const string mrgSegmentId = "MRG";
            const string invalidMergeFromPatientNameIndicator = "MERGING INTO";
            const string detailedErrorMessageKey = "DetailedErrorMessage";
            const string acknowledgementErrorMessageKey = "AcknowledgementErrorMessage";

            if (hl7Message == null) { throw (new ArgumentNullException("hl7Message")); }
            //
            Hashtable errorMessageTable = null;
            // 
            var segmentTable = ParseGetMessageSegments(hl7Message);
            //
            if (segmentTable != null && segmentTable.Count > 0)
            {
                var delimiters = hl7Message.ToCharArray(3, 5);
                //
                var pidSegment = segmentTable[pidSegmentId] != null ? segmentTable[pidSegmentId].ToString() : null;
                //
                if (pidSegment != null)
                {
                    var patientName = pidSegment.Split(delimiters[0])[5].Split(delimiters[1]);
                    //
                    if (patientName.Length > 0)
                    {
                        var patientLastName = patientName.Length >= 1 ? HL7Utility.ConvertString(patientName[0]) : null;
                        var patientFirstName = patientName.Length >= 2 ? HL7Utility.ConvertString(patientName[1]) : null;
                        var patientMiddleName = patientName.Length >= 3 ? HL7Utility.ConvertString(patientName[2]) : null;
                        //
                        ValidatePatientName(patientLastName, patientFirstName, patientMiddleName, ref errorMessageTable);
                    }
                }
                else
                {
                    // No PID segment found
                    ValidatePatientName(null, null, null, ref errorMessageTable);
                }
                //
                if (segmentTable.ContainsKey(mrgSegmentId))
                {
                    var mrgSegment = segmentTable[mrgSegmentId] != null ? segmentTable[mrgSegmentId].ToString() : null;
                    //
                    if (mrgSegment != null)
                    {
                        // CR 2931
                        var mergeFromPatientNameFullString = mrgSegment.Split(delimiters[0])[7];
                        var mergeFromPatientNameComponents = mergeFromPatientNameFullString.Split(delimiters[1]);
                        //
                        // CR 2931: if merge-from name contains the invalid indicator, we ignore the length requirement; we won't be using this name anyway
                        if (mergeFromPatientNameFullString.IndexOf(invalidMergeFromPatientNameIndicator, StringComparison.Ordinal) == -1 && mergeFromPatientNameComponents.Length > 0)
                        {
                            var mergeFromPatientLastName = mergeFromPatientNameComponents.Length >= 1 ? HL7Utility.ConvertString(mergeFromPatientNameComponents[0]) : null;
                            var mergeFromPatientFirstName = mergeFromPatientNameComponents.Length >= 2 ? HL7Utility.ConvertString(mergeFromPatientNameComponents[1]) : null;
                            var mergeFromPatientMiddleName = mergeFromPatientNameComponents.Length >= 3 ? HL7Utility.ConvertString(mergeFromPatientNameComponents[2]) : null;
                            //
                            ValidatePatientName(mergeFromPatientLastName, mergeFromPatientFirstName, mergeFromPatientMiddleName, ref errorMessageTable);
                        }
                    }
                    else
                    {
                        // Merge message but with no MRG segment found
                        ValidatePatientName(null, null, null, ref errorMessageTable);
                    }
                }
                //
                // Clean up the table then return
                if (errorMessageTable != null)
                {
                    var acknowledgementErrorList = new ArrayList();
                    var acknowledgementErrorMessage = new StringBuilder();
                    //
                    foreach (string key in errorMessageTable.Keys)
                    {
                        if (key != detailedErrorMessageKey)
                        {
                            var keyValue = errorMessageTable[key];
                            //
                            if (!acknowledgementErrorList.Contains(keyValue))
                            {
                                acknowledgementErrorList.Add(keyValue);
                                if (acknowledgementErrorMessage.Length > 0) { acknowledgementErrorMessage.Append(" "); }
                                acknowledgementErrorMessage.Append(keyValue);
                            }
                        }
                    }
                    //
                    var detailedErrorMessage = errorMessageTable.ContainsKey(detailedErrorMessageKey) ? errorMessageTable[detailedErrorMessageKey].ToString() : null;
                    errorMessageTable = new Hashtable
                        {
                            {acknowledgementErrorMessageKey, acknowledgementErrorMessage.ToString()},
                            {detailedErrorMessageKey, detailedErrorMessage}
                        };
                }
            }
            //
            return errorMessageTable;
        }

        /// <summary>
        /// CR 2606: Verifies the PID segment required patient name fields are present 
        /// and all fields included in the message fit within the max allowed by VBECS; 
        /// returns error message(s) if any name field or the entire name length exceeds 
        /// the max allowed value
        /// 
        /// CR 2659: Simplified to just validate a patient's name so that it might be
        /// called multiple times (ex. merge for PID and MRG segments)
        /// 
        /// CR 2665: Changed last and first name existence check to an OR condition;
        /// updated to use new error message
        /// </summary>
        private static void ValidatePatientName(string lastName, string firstName, string middleName, ref Hashtable errorMessageTable)
        {
            const string patientNameNotFound = "The Patient's Last Name or First Name was not found in the HL7 Message.";
            const string patientNameFieldSizeNotSupported = "The Patient's Name field size exceeds VBECS maximum supported value.";
            // Read port number from configuration

            var maxPatientLastNameLength = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientLastNameMaximumLength", typeof(int));
            var fullNameFieldLength = 0;
            //
            StringBuilder detailedErrorMessage = null;
            //
            if (errorMessageTable != null && errorMessageTable.ContainsKey("DetailedErrorMessage"))
            {
                detailedErrorMessage = new StringBuilder(errorMessageTable["DetailedErrorMessage"].ToString());
            }
            // **********************************************************************************************
            // Verify patient last name and first name are present in message
            // CR 2665
            if (lastName == null || firstName == null)
            {
                if (errorMessageTable == null) { errorMessageTable = new Hashtable(); }
                //
                const string key = "AcknowledgementErrorMessage.PATIENT_NAME_NOT_FOUND";
                //
                if (!errorMessageTable.ContainsKey(key))
                {
                    errorMessageTable.Add(key, patientNameNotFound);
                    //
                    if (detailedErrorMessage == null) { detailedErrorMessage = new StringBuilder(patientNameNotFound); }
                    else { detailedErrorMessage.Append("\n").Append(patientNameNotFound); }
                }
            }
            else
            {
                // **********************************************************************************************
                // Verify patient's last name field size
                //
                var lastNameLength = lastName.Length;
                //
                fullNameFieldLength += lastNameLength;
                //
                if (lastNameLength > maxPatientLastNameLength)
                {
                    if (errorMessageTable == null) { errorMessageTable = new Hashtable(); }
                    //
                    const string key = "AcknowledgementErrorMessage.PATIENT_LAST_NAME_FIELD_SIZE_NOT_SUPPORTED";
                    //
                    if (!errorMessageTable.ContainsKey(key))
                    {
                        errorMessageTable.Add(key, patientNameFieldSizeNotSupported);
                        //
                        if (detailedErrorMessage == null) { detailedErrorMessage = new StringBuilder(); }
                        else { detailedErrorMessage.Append("\n"); }
                        //
                        detailedErrorMessage.Append("The Patient's Last Name field size (");
                        detailedErrorMessage.Append(lastNameLength);
                        detailedErrorMessage.Append(") exceeds VBECS maximum supported value (");
                        detailedErrorMessage.Append(maxPatientLastNameLength);
                        detailedErrorMessage.Append(" characters).");
                    }
                }
                // **********************************************************************************************
                // Verify patient's first name field size
                //
                var firstNameLength = firstName.Length;
                //
                // The extra 1 here accounts for the comma in VistA
                fullNameFieldLength += (firstNameLength + 1);
                //
                if (firstNameLength > maxPatientLastNameLength)
                {
                    if (errorMessageTable == null) { errorMessageTable = new Hashtable(); }
                    //
                    const string key = "AcknowledgementErrorMessage.PATIENT_FIRST_NAME_FIELD_SIZE_NOT_SUPPORTED";
                    //
                    if (!errorMessageTable.ContainsKey(key))
                    {
                        errorMessageTable.Add(key, patientNameFieldSizeNotSupported);
                        //
                        if (detailedErrorMessage == null) { detailedErrorMessage = new StringBuilder(); }
                        else { detailedErrorMessage.Append("\n"); }
                        //
                        detailedErrorMessage.Append("The Patient's First Name field size (");
                        detailedErrorMessage.Append(firstNameLength);
                        detailedErrorMessage.Append(") exceeds VBECS maximum supported value (");
                        detailedErrorMessage.Append(maxPatientLastNameLength);
                        detailedErrorMessage.Append(" characters).");
                    }
                }
                // **********************************************************************************************
                // Verify patient's middle name field size
                //
                if (middleName != null)
                {
                    var middleNameLength = middleName.Length;
                    //
                    // The extra 1 here accounts for the space in VistA
                    fullNameFieldLength += (middleNameLength + 1);
                    //
                    if (middleNameLength > maxPatientLastNameLength)
                    {
                        if (errorMessageTable == null) { errorMessageTable = new Hashtable(); }
                        //
                        const string key = "AcknowledgementErrorMessage.PATIENT_MIDDLE_NAME_FIELD_SIZE_NOT_SUPPORTED";
                        //
                        if (!errorMessageTable.ContainsKey(key))
                        {
                            errorMessageTable.Add(key, patientNameFieldSizeNotSupported);
                            //
                            if (detailedErrorMessage == null) { detailedErrorMessage = new StringBuilder(); }
                            else { detailedErrorMessage.Append("\n"); }
                            //
                            detailedErrorMessage.Append("The Patient's Middle Name field size (");
                            detailedErrorMessage.Append(middleNameLength);
                            detailedErrorMessage.Append(") exceeds VBECS maximum supported value (");
                            detailedErrorMessage.Append(maxPatientLastNameLength);
                            detailedErrorMessage.Append(" characters).");
                        }
                    }
                }
                // **********************************************************************************************
                // Verify total name length
                //
                var maxPatientFullNameLength = (int)GlobalContext.Instance().AppSettingsReader.GetValue("PatientFullNameMaximumLength", typeof(int));
                if (fullNameFieldLength > maxPatientFullNameLength)
                {
                    if (errorMessageTable == null) { errorMessageTable = new Hashtable(); }
                    //
                    const string key = "AcknowledgementErrorMessage.PATIENT_FULL_NAME_FIELD_SIZE_NOT_SUPPORTED";
                    //
                    if (!errorMessageTable.ContainsKey(key))
                    {
                        errorMessageTable.Add(key, patientNameFieldSizeNotSupported);
                        //
                        if (detailedErrorMessage == null) { detailedErrorMessage = new StringBuilder(); }
                        else { detailedErrorMessage.Append("\n"); }
                        //
                        detailedErrorMessage.Append("The Patient's Full Name field size (");
                        detailedErrorMessage.Append(fullNameFieldLength);
                        detailedErrorMessage.Append(") exceeds VBECS maximum supported value (");
                        detailedErrorMessage.Append(maxPatientFullNameLength);
                        detailedErrorMessage.Append(" characters).");
                    }
                }
            }
            //
            if (detailedErrorMessage != null && errorMessageTable != null)
            {
                if (errorMessageTable.ContainsKey("DetailedErrorMessage"))
                {
                    errorMessageTable["DetailedErrorMessage"] = detailedErrorMessage;
                }
                else
                {
                    errorMessageTable.Add("DetailedErrorMessage", detailedErrorMessage);
                }
            }
        }

        /// <summary>
        /// CR 2659: returns desired HL7 message segments (ex. used to get the PID and MRG from 
        /// patient merge messages)
        /// </summary>
        private static Hashtable ParseGetMessageSegments(string hl7Message)
        {
            const string pidSegmentId = "PID";
            const string mrgSegmentId = "MRG";
            const string mergeMessageType = "MPIF TRIGGER";

            if (hl7Message == null)
            {
                throw (new ArgumentNullException("hl7Message"));
            }
            //
            var segments = hl7Message.Split(new[] { '\r' });
            //
            var returnCount = segments[0].IndexOf(mergeMessageType, StringComparison.Ordinal) > -1 ? 2 : 1;
            var updatedCount = 0;
            //
            var segmentTable = new Hashtable(returnCount) {{pidSegmentId, null}};
            if (returnCount == 2) { segmentTable.Add(mrgSegmentId, null); }
            //
            for (var i = 0; i <= segments.Length - 1; i++)
            {
                if (segments[i] == null || segments[i].Length <= 3) continue;
                var segmentId = segments[i].Substring(0, 3);
                //
                // If this is a merge message we need to check the PID and MRG segments
                if (segmentId == pidSegmentId || (returnCount == 2 && segmentId == mrgSegmentId))
                {
                    segmentTable[segmentId] = segments[i];
                    updatedCount += 1;
                    //
                    if (updatedCount == returnCount) { break; }
                }
            }
            //
            return segmentTable;
        }

        /// <summary>
        /// Implements VistA Application Interface Specification: 8.1 (unsupported patient names)
        /// CR 2606
        /// </summary>
        private HL7ProtocolMessage HandleUnsupportedPatientName(HL7Interface hl7Interface, string hl7Message, Hashtable errorMessageTable, string ipAddress)
        {
            const string acknowledgementErrorMessageKey = "AcknowledgementErrorMessage";
            const string detailedErrorMessageKey = "DetailedErrorMessage";
            const string bceCotsInterface = "BCE_COTS";
            const string cprsInterface = "CPRS";
            const string patientMergeInterface = "Patient_Merge";
            const string patientUpdateInterface = "Patient_Update";



            // **********************************************************************************************
            // CR 2606: If patient name length exceeds the max allowed value then write error to event log, 
            // email the interface administrator and return an acknowledgement reject (AR) message for the 
            // CPRS or Patient Update interfaces and an application accept (AA) message for the Patient Merge 
            // interface, which cannot process reject messages at this time
            //
            HL7ProtocolMessage ackMessage = null;
            //
            // Changed keys to refer to string constants (which were added for CR 2659)
            var acknowledgementErrorMessage = errorMessageTable.ContainsKey(acknowledgementErrorMessageKey) ? errorMessageTable[acknowledgementErrorMessageKey].ToString() : string.Empty;
            var detailedErrorMessage = errorMessageTable.ContainsKey(detailedErrorMessageKey) ? errorMessageTable[detailedErrorMessageKey].ToString() : string.Empty;
            //
            switch (hl7Interface.InterfaceName)
            {
                case bceCotsInterface:
                case cprsInterface:
                case patientUpdateInterface:
                    {
                        //HL7OmgMessage message = new HL7OmgMessage( hl7Message );
                        //ackMessage = CprsOrgMessage.AckError( message, AckCodes.AR, acknowledgementErrorMessage );
                        ackMessage = HL7AckMessage.BuildAckMessage(hl7Message, AckCodes.AR, acknowledgementErrorMessage);
                        break;
                    }
                case patientMergeInterface:
                    {
                        _logger.Info("PatientMerge sends AA ACK but it failed with: " + acknowledgementErrorMessage);
                        ackMessage = HL7AckMessage.BuildAckMessage(hl7Message, AckCodes.AA, string.Empty);
                        break;
                    }
            }
            //
            var errorMsg = send_unsupported_Hl7_nessage_error_details(ipAddress, hl7Message, detailedErrorMessage, hl7Interface);
            // EventLogAppender
            // Record full error details including HL7 message contents in the event log
            _eventsLogger.Error(errorMsg);
            //
            return ackMessage;
        }
    }
}
